# Fierce Mobile Game API - Unity Integration Guide

## Overview
This API system provides a complete backend solution for your 3D Race game with user accounts, wallet system, withdrawal management, and AdMob integration.

## API Base URL
```
http://localhost/fierce-mobile-api/api/
```

## Unity Integration

### 1. User Authentication
```csharp
using System.Collections;
using UnityEngine;
using UnityEngine.Networking;

[System.Serializable]
public class AuthRequest
{
    public string device_id;
    public string username;
    public string email;
}

[System.Serializable]
public class AuthResponse
{
    public bool success;
    public string message;
    public UserData user;
}

[System.Serializable]
public class UserData
{
    public int id;
    public string username;
    public string email;
    public int total_playtime_minutes;
    public float current_balance;
    public float total_earned;
    public float total_withdrawn;
    public float admob_earnings;
    public string status;
}

public class GameAPI : MonoBehaviour
{
    private const string API_BASE_URL = "http://localhost/fierce-mobile-api/api/";
    private UserData currentUser;
    private int currentSessionId;

    void Start()
    {
        AuthenticateUser();
    }

    public void AuthenticateUser()
    {
        string deviceId = SystemInfo.deviceUniqueIdentifier;
        StartCoroutine(AuthenticateCoroutine(deviceId));
    }

    IEnumerator AuthenticateCoroutine(string deviceId)
    {
        AuthRequest authData = new AuthRequest
        {
            device_id = deviceId,
            username = "Player_" + deviceId.Substring(0, 8),
            email = deviceId + "@game.local"
        };

        string jsonData = JsonUtility.ToJson(authData);
        
        using (UnityWebRequest request = UnityWebRequest.Post(API_BASE_URL + "auth.php", jsonData, "application/json"))
        {
            yield return request.SendWebRequest();

            if (request.result == UnityWebRequest.Result.Success)
            {
                AuthResponse response = JsonUtility.FromJson<AuthResponse>(request.downloadHandler.text);
                if (response.success)
                {
                    currentUser = response.user;
                    Debug.Log($"User authenticated: {currentUser.username}, Balance: ${currentUser.current_balance}");
                    StartGameSession();
                }
            }
            else
            {
                Debug.LogError("Authentication failed: " + request.error);
            }
        }
    }
}
```

### 2. Game Session Management
```csharp
[System.Serializable]
public class SessionRequest
{
    public int user_id;
}

[System.Serializable]
public class SessionResponse
{
    public bool success;
    public string message;
    public int session_id;
}

[System.Serializable]
public class EndSessionRequest
{
    public int session_id;
    public bool race_completed;
}

public void StartGameSession()
{
    if (currentUser != null)
    {
        StartCoroutine(StartSessionCoroutine());
    }
}

IEnumerator StartSessionCoroutine()
{
    SessionRequest sessionData = new SessionRequest
    {
        user_id = currentUser.id
    };

    string jsonData = JsonUtility.ToJson(sessionData);
    
    using (UnityWebRequest request = UnityWebRequest.Post(API_BASE_URL + "start_session.php", jsonData, "application/json"))
    {
        yield return request.SendWebRequest();

        if (request.result == UnityWebRequest.Result.Success)
        {
            SessionResponse response = JsonUtility.FromJson<SessionResponse>(request.downloadHandler.text);
            if (response.success)
            {
                currentSessionId = response.session_id;
                Debug.Log($"Session started: {currentSessionId}");
            }
        }
    }
}

public void EndGameSession(bool raceCompleted = false)
{
    if (currentSessionId > 0)
    {
        StartCoroutine(EndSessionCoroutine(raceCompleted));
    }
}

IEnumerator EndSessionCoroutine(bool raceCompleted)
{
    EndSessionRequest endData = new EndSessionRequest
    {
        session_id = currentSessionId,
        race_completed = raceCompleted
    };

    string jsonData = JsonUtility.ToJson(endData);
    
    using (UnityWebRequest request = UnityWebRequest.Post(API_BASE_URL + "end_session.php", jsonData, "application/json"))
    {
        yield return request.SendWebRequest();

        if (request.result == UnityWebRequest.Result.Success)
        {
            var response = JsonUtility.FromJson<dynamic>(request.downloadHandler.text);
            Debug.Log("Session ended successfully");
            currentSessionId = 0;
        }
    }
}
```

### 3. AdMob Integration
```csharp
[System.Serializable]
public class AdMobRequest
{
    public int user_id;
    public string ad_type;
    public float revenue;
    public string ad_unit_id;
}

public void RecordAdMobRevenue(string adType, float revenue, string adUnitId = null)
{
    if (currentUser != null)
    {
        StartCoroutine(RecordAdMobCoroutine(adType, revenue, adUnitId));
    }
}

IEnumerator RecordAdMobCoroutine(string adType, float revenue, string adUnitId)
{
    AdMobRequest adData = new AdMobRequest
    {
        user_id = currentUser.id,
        ad_type = adType,
        revenue = revenue,
        ad_unit_id = adUnitId
    };

    string jsonData = JsonUtility.ToJson(adData);
    
    using (UnityWebRequest request = UnityWebRequest.Post(API_BASE_URL + "record_admob.php", jsonData, "application/json"))
    {
        yield return request.SendWebRequest();

        if (request.result == UnityWebRequest.Result.Success)
        {
            Debug.Log("AdMob revenue recorded successfully");
        }
    }
}

// Example usage with Google Mobile Ads SDK
public void OnRewardedAdComplete()
{
    // Record rewarded ad revenue (estimate $0.01 per view)
    RecordAdMobRevenue("rewarded", 0.01f, "your-rewarded-ad-unit-id");
}

public void OnInterstitialAdComplete()
{
    // Record interstitial ad revenue (estimate $0.005 per view)
    RecordAdMobRevenue("interstitial", 0.005f, "your-interstitial-ad-unit-id");
}
```

### 4. Withdrawal System
```csharp
[System.Serializable]
public class WithdrawalRequest
{
    public int user_id;
    public float amount;
    public string payment_type;
    public string payment_address;
}

public void RequestWithdrawal(float amount, string paymentType, string paymentAddress)
{
    StartCoroutine(WithdrawalCoroutine(amount, paymentType, paymentAddress));
}

IEnumerator WithdrawalCoroutine(float amount, string paymentType, string paymentAddress)
{
    WithdrawalRequest withdrawalData = new WithdrawalRequest
    {
        user_id = currentUser.id,
        amount = amount,
        payment_type = paymentType,
        payment_address = paymentAddress
    };

    string jsonData = JsonUtility.ToJson(withdrawalData);
    
    using (UnityWebRequest request = UnityWebRequest.Post(API_BASE_URL + "withdrawal.php", jsonData, "application/json"))
    {
        yield return request.SendWebRequest();

        if (request.result == UnityWebRequest.Result.Success)
        {
            var response = JsonUtility.FromJson<dynamic>(request.downloadHandler.text);
            Debug.Log("Withdrawal request submitted");
        }
    }
}
```

## Application Events

### Game Events to Track
1. **Game Start**: Call `StartGameSession()`
2. **Race Complete**: Call `EndGameSession(true)`
3. **Game Exit**: Call `EndGameSession(false)`
4. **Ad Viewed**: Call `RecordAdMobRevenue()`
5. **Withdrawal Request**: Call `RequestWithdrawal()`

### Example Implementation
```csharp
public class RaceManager : MonoBehaviour
{
    public GameAPI gameAPI;
    
    void OnRaceStart()
    {
        // Race started - session already active
    }
    
    void OnRaceComplete()
    {
        // Race completed successfully
        gameAPI.EndGameSession(true);
        
        // Show rewarded ad for bonus
        ShowRewardedAd();
    }
    
    void OnGameExit()
    {
        // Player exiting game
        gameAPI.EndGameSession(false);
    }
    
    void ShowRewardedAd()
    {
        // Show rewarded ad and on complete:
        gameAPI.RecordAdMobRevenue("rewarded", 0.01f);
    }
}
```

## Admin Panel Access

Access the admin panel at:
```
http://localhost/fierce-mobile-api/admin/
```

Default admin credentials:
- Username: admin
- Password: admin123

## Database Setup

1. Import the database schema:
```sql
mysql -u anemal_fierce -p < database/schema.sql
```

2. Update database connection in `config/database.php`

## Security Notes

1. **API Authentication**: Implement proper API keys for production
2. **Input Validation**: All inputs are sanitized
3. **SQL Injection Protection**: Using PDO with prepared statements
4. **XSS Protection**: HTML special characters are escaped

## Earning Rates Configuration

Default settings (configurable via admin panel):
- Earning rate: $0.10 per minute of gameplay
- Minimum withdrawal: $5.00
- Daily earning limit: $50.00
- AdMob revenue share: 70% to users
- Race completion bonus: 50% extra earnings

## Support

For any issues or questions, check the admin panel logs and ensure:
1. Database connection is working
2. XAMPP/Apache is running
3. PHP modules are enabled (PDO, MySQL)
4. File permissions are correct
