<?php
// Note: Database class is injected via constructor, no direct require needed
// require_once '../config/database.php';

class UserSession
{
    private $conn;
    private $table_name = "user_sessions";

    public $id;
    public $user_id;
    public $session_start;
    public $session_end;
    public $duration_minutes;
    public $earnings_generated;
    public $admob_revenue;
    public $race_completed;

    public function __construct($db)
    {
        $this->conn = $db;
    }

    // Start new session
    public function startSession($user_id)
    {
        $query = "INSERT INTO " . $this->table_name . " (user_id) VALUES (:user_id)";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);

        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            $this->user_id = $user_id;
            return $this->id;
        }
        return false;
    }

    // End session and calculate earnings
    public function endSession($session_id, $race_completed = false)
    {
        // Get session start time
        $query = "SELECT session_start, user_id FROM " . $this->table_name . " WHERE id = :session_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':session_id', $session_id);
        $stmt->execute();

        $session = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$session) return false;

        // Calculate duration
        $start_time = new DateTime($session['session_start']);
        $end_time = new DateTime();
        $duration_minutes = $end_time->diff($start_time)->i + ($end_time->diff($start_time)->h * 60);

        // Calculate earnings based on playtime
        $earning_rate = $this->getSystemSetting('earning_rate_per_minute');
        $earnings = $duration_minutes * $earning_rate;

        // Bonus for completed races
        if ($race_completed) {
            $earnings *= 1.5; // 50% bonus for completing races
        }

        // Check daily earning limit
        $daily_limit = $this->getSystemSetting('daily_earning_limit');
        $today_earnings = $this->getTodayEarnings($session['user_id']);

        if ($today_earnings + $earnings > $daily_limit) {
            $earnings = max(0, $daily_limit - $today_earnings);
        }

        // Update session
        $query = "UPDATE " . $this->table_name . " 
                  SET session_end = CURRENT_TIMESTAMP, 
                      duration_minutes = :duration, 
                      earnings_generated = :earnings,
                      race_completed = :race_completed
                  WHERE id = :session_id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':duration', $duration_minutes);
        $stmt->bindParam(':earnings', $earnings);
        $stmt->bindParam(':race_completed', $race_completed, PDO::PARAM_BOOL);
        $stmt->bindParam(':session_id', $session_id);

        if ($stmt->execute()) {
            // Update user balance and playtime
            $this->updateUserStats($session['user_id'], $duration_minutes, $earnings);
            return [
                'duration_minutes' => $duration_minutes,
                'earnings' => $earnings,
                'daily_earnings' => $today_earnings + $earnings
            ];
        }

        return false;
    }

    // Record AdMob revenue
    public function recordAdMobRevenue($user_id, $ad_type, $revenue, $ad_unit_id = null)
    {
        // Insert into admob_earnings table
        $query = "INSERT INTO admob_earnings (user_id, ad_type, revenue, ad_unit_id) 
                  VALUES (:user_id, :ad_type, :revenue, :ad_unit_id)";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':ad_type', $ad_type);
        $stmt->bindParam(':revenue', $revenue);
        $stmt->bindParam(':ad_unit_id', $ad_unit_id);

        if ($stmt->execute()) {
            // Calculate user's share (default 70%)
            $revenue_share = $this->getSystemSetting('admob_revenue_share') / 100;
            $user_earnings = $revenue * $revenue_share;

            // Update user's balance and admob earnings
            $query = "UPDATE users 
                      SET current_balance = current_balance + :earnings,
                          total_earned = total_earned + :earnings,
                          admob_earnings = admob_earnings + :earnings
                      WHERE id = :user_id";

            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':earnings', $user_earnings);
            $stmt->bindParam(':user_id', $user_id);

            return $stmt->execute();
        }

        return false;
    }

    private function getTodayEarnings($user_id)
    {
        $query = "SELECT COALESCE(SUM(earnings_generated), 0) as today_earnings 
                  FROM " . $this->table_name . " 
                  WHERE user_id = :user_id AND DATE(session_start) = CURDATE()";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? floatval($result['today_earnings']) : 0;
    }

    private function updateUserStats($user_id, $duration_minutes, $earnings)
    {
        $query = "UPDATE users 
                  SET total_playtime_minutes = total_playtime_minutes + :duration,
                      current_balance = current_balance + :earnings,
                      total_earned = total_earned + :earnings
                  WHERE id = :user_id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':duration', $duration_minutes);
        $stmt->bindParam(':earnings', $earnings);
        $stmt->bindParam(':user_id', $user_id);

        return $stmt->execute();
    }

    private function getSystemSetting($key)
    {
        $query = "SELECT setting_value FROM system_settings WHERE setting_key = :key";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':key', $key);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? floatval($result['setting_value']) : 0;
    }
}
