<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../../config/database.php';
require_once '../../models/User.php';
require_once '../../models/WithdrawalRequest.php';

try {
    $database = new Database();
    $db = $database->getConnection();

    if (!$db) {
        throw new Exception('Database connection failed');
    }

    // Get user statistics
    $userStats = User::getUserStats($db);

    // Get withdrawal statistics
    $withdrawalRequest = new WithdrawalRequest($db);
    $withdrawalStats = $withdrawalRequest->getStats();

    // Get chart data for last 7 days
    $chartData = getChartData($db);

    $stats = [
        'total_users' => $userStats['total_users'] ?? 0,
        'pending_withdrawals' => $withdrawalStats['pending_requests'] ?? 0,
        'total_payouts' => number_format($withdrawalStats['total_completed_amount'] ?? 0, 2),
        'total_balance' => number_format($userStats['total_balance'] ?? 0, 2)
    ];

    echo json_encode([
        'success' => true,
        'stats' => $stats,
        'chart_data' => $chartData
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error fetching dashboard data: ' . $e->getMessage(),
        'error_details' => $e->getTraceAsString()
    ]);
}

function getChartData($db)
{
    try {
        // Get earnings data for last 7 days
        $earningsQuery = "SELECT DATE(session_start) as date, SUM(earnings_generated) as total_earnings 
                          FROM user_sessions 
                          WHERE session_start >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                          GROUP BY DATE(session_start)
                          ORDER BY date";

        $stmt = $db->prepare($earningsQuery);
        $stmt->execute();
        $earningsData = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Get withdrawal status data
        $withdrawalQuery = "SELECT status, COUNT(*) as count 
                            FROM withdrawal_requests 
                            GROUP BY status";

        $stmt = $db->prepare($withdrawalQuery);
        $stmt->execute();
        $withdrawalData = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Format data for charts
        $earnings_labels = [];
        $earnings_values = [];
        $withdrawal_status = [0, 0, 0, 0]; // pending, approved, completed, rejected

        // Process earnings data
        foreach ($earningsData as $row) {
            $earnings_labels[] = date('M d', strtotime($row['date']));
            $earnings_values[] = floatval($row['total_earnings']);
        }

        // Process withdrawal data
        $statusMap = ['pending' => 0, 'approved' => 1, 'completed' => 2, 'rejected' => 3];
        foreach ($withdrawalData as $row) {
            if (isset($statusMap[$row['status']])) {
                $withdrawal_status[$statusMap[$row['status']]] = intval($row['count']);
            }
        }

        return [
            'earnings_labels' => $earnings_labels,
            'earnings_data' => $earnings_values,
            'withdrawal_status' => $withdrawal_status
        ];
    } catch (Exception $e) {
        // Return empty data if chart data fails
        return [
            'earnings_labels' => [],
            'earnings_data' => [],
            'withdrawal_status' => [0, 0, 0, 0]
        ];
    }
}
