<?php
// Note: Database class is injected via constructor, no direct require needed
// require_once '../config/database.php';

class WithdrawalRequest
{
    private $conn;
    private $table_name = "withdrawal_requests";

    public $id;
    public $user_id;
    public $amount;
    public $payment_type;
    public $payment_address;
    public $status;
    public $admin_notes;
    public $processed_by;
    public $request_date;
    public $processed_date;

    public function __construct($db)
    {
        $this->conn = $db;
    }

    // Create withdrawal request
    public function create()
    {
        // Check minimum withdrawal amount
        $min_withdrawal = $this->getSystemSetting('minimum_withdrawal');
        if ($this->amount < $min_withdrawal) {
            return ['success' => false, 'message' => "Minimum withdrawal amount is $" . $min_withdrawal];
        }

        // Check user balance
        if (!$this->checkUserBalance()) {
            return ['success' => false, 'message' => 'Insufficient balance'];
        }

        $query = "INSERT INTO " . $this->table_name . " 
                  SET user_id=:user_id, amount=:amount, payment_type=:payment_type, 
                      payment_address=:payment_address";

        $stmt = $this->conn->prepare($query);

        // Sanitize
        $this->payment_address = htmlspecialchars(strip_tags($this->payment_address));

        // Bind values
        $stmt->bindParam(":user_id", $this->user_id);
        $stmt->bindParam(":amount", $this->amount);
        $stmt->bindParam(":payment_type", $this->payment_type);
        $stmt->bindParam(":payment_address", $this->payment_address);

        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();

            // Deduct amount from user balance
            $this->deductFromUserBalance();

            return ['success' => true, 'message' => 'Withdrawal request submitted successfully', 'request_id' => $this->id];
        }

        return ['success' => false, 'message' => 'Failed to submit withdrawal request'];
    }

    // Get all withdrawal requests with user info
    public function getAllWithUserInfo($offset = 0, $limit = 20, $status_filter = null)
    {
        $where_clause = "";
        if ($status_filter) {
            $where_clause = "WHERE wr.status = :status_filter";
        }

        $query = "SELECT wr.*, u.username, u.email, u.current_balance,
                         a.username as processed_by_name
                  FROM " . $this->table_name . " wr
                  LEFT JOIN users u ON wr.user_id = u.id
                  LEFT JOIN admins a ON wr.processed_by = a.id
                  " . $where_clause . "
                  ORDER BY wr.request_date DESC
                  LIMIT :limit OFFSET :offset";

        $stmt = $this->conn->prepare($query);

        if ($status_filter) {
            $stmt->bindParam(':status_filter', $status_filter);
        }

        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Update withdrawal status
    public function updateStatus($new_status, $admin_id, $admin_notes = '')
    {
        $query = "UPDATE " . $this->table_name . " 
                  SET status = :status, processed_by = :admin_id, 
                      admin_notes = :admin_notes, processed_date = CURRENT_TIMESTAMP
                  WHERE id = :id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':status', $new_status);
        $stmt->bindParam(':admin_id', $admin_id);
        $stmt->bindParam(':admin_notes', $admin_notes);
        $stmt->bindParam(':id', $this->id);

        if ($stmt->execute()) {
            // If rejected, refund the amount to user
            if ($new_status === 'rejected') {
                $this->refundToUser();
            }

            // If completed, update user withdrawn total
            if ($new_status === 'completed') {
                $this->updateUserWithdrawnTotal();
            }

            return true;
        }

        return false;
    }

    // Get withdrawal statistics
    public function getStats()
    {
        $query = "SELECT 
                    COUNT(*) as total_requests,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_requests,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_requests,
                    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_requests,
                    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_requests,
                    SUM(CASE WHEN status = 'completed' THEN amount ELSE 0 END) as total_completed_amount,
                    SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END) as total_pending_amount
                  FROM " . $this->table_name;

        $stmt = $this->conn->prepare($query);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get user's withdrawal requests
    public function getUserRequests($user_id, $limit = 10)
    {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE user_id = :user_id 
                  ORDER BY request_date DESC 
                  LIMIT :limit";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    private function checkUserBalance()
    {
        $query = "SELECT current_balance FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $this->user_id);
        $stmt->execute();

        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        return $user && $user['current_balance'] >= $this->amount;
    }

    private function deductFromUserBalance()
    {
        $query = "UPDATE users SET current_balance = current_balance - :amount WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':amount', $this->amount);
        $stmt->bindParam(':user_id', $this->user_id);
        $stmt->execute();
    }

    private function refundToUser()
    {
        $query = "UPDATE users SET current_balance = current_balance + :amount WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':amount', $this->amount);
        $stmt->bindParam(':user_id', $this->user_id);
        $stmt->execute();
    }

    private function updateUserWithdrawnTotal()
    {
        $query = "UPDATE users SET total_withdrawn = total_withdrawn + :amount WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':amount', $this->amount);
        $stmt->bindParam(':user_id', $this->user_id);
        $stmt->execute();
    }

    private function getSystemSetting($key)
    {
        $query = "SELECT setting_value FROM system_settings WHERE setting_key = :key";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':key', $key);
        $stmt->execute();

        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? floatval($result['setting_value']) : 0;
    }
}
