using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Networking;

namespace FierceGame
{
    [System.Serializable]
    public class AuthRequest
    {
        public string device_id;
        public string username;
        public string email;
    }

    [System.Serializable]
    public class AuthResponse
    {
        public bool success;
        public string message;
        public UserData user;
    }

    [System.Serializable]
    public class UserData
    {
        public int id;
        public string username;
        public string email;
        public int total_playtime_minutes;
        public float current_balance;
        public float total_earned;
        public float total_withdrawn;
        public float admob_earnings;
        public string status;
    }

    [System.Serializable]
    public class SessionRequest
    {
        public int user_id;
    }

    [System.Serializable]
    public class SessionResponse
    {
        public bool success;
        public string message;
        public int session_id;
    }

    [System.Serializable]
    public class EndSessionRequest
    {
        public int session_id;
        public bool race_completed;
    }

    [System.Serializable]
    public class EndSessionResponse
    {
        public bool success;
        public string message;
        public SessionData session_data;
    }

    [System.Serializable]
    public class SessionData
    {
        public int duration_minutes;
        public float earnings;
        public float daily_earnings;
    }

    [System.Serializable]
    public class AdMobRequest
    {
        public int user_id;
        public string ad_type;
        public float revenue;
        public string ad_unit_id;
    }

    [System.Serializable]
    public class WithdrawalRequest
    {
        public int user_id;
        public float amount;
        public string payment_type;
        public string payment_address;
    }

    [System.Serializable]
    public class ApiResponse
    {
        public bool success;
        public string message;
    }

    public class FierceGameAPI : MonoBehaviour
    {
        [Header("API Configuration")]
        [SerializeField] private string apiBaseUrl = "http://localhost/fierce-mobile-api/api/";
        
        [Header("Events")]
        public UnityEngine.Events.UnityEvent<UserData> OnUserAuthenticated;
        public UnityEngine.Events.UnityEvent<SessionData> OnSessionEnded;
        public UnityEngine.Events.UnityEvent<string> OnError;

        private UserData currentUser;
        private int currentSessionId;
        private bool isSessionActive = false;

        public UserData CurrentUser => currentUser;
        public bool IsSessionActive => isSessionActive;

        void Start()
        {
            AuthenticateUser();
        }

        void OnApplicationPause(bool pauseStatus)
        {
            if (pauseStatus && isSessionActive)
            {
                EndGameSession(false);
            }
            else if (!pauseStatus && currentUser != null)
            {
                StartGameSession();
            }
        }

        void OnApplicationFocus(bool hasFocus)
        {
            if (!hasFocus && isSessionActive)
            {
                EndGameSession(false);
            }
        }

        #region Authentication

        public void AuthenticateUser()
        {
            string deviceId = SystemInfo.deviceUniqueIdentifier;
            StartCoroutine(AuthenticateCoroutine(deviceId));
        }

        private IEnumerator AuthenticateCoroutine(string deviceId)
        {
            AuthRequest authData = new AuthRequest
            {
                device_id = deviceId,
                username = "Player_" + deviceId.Substring(0, 8),
                email = deviceId + "@fiercegame.local"
            };

            yield return StartCoroutine(SendPostRequest<AuthResponse>("auth.php", authData, (response) =>
            {
                if (response.success)
                {
                    currentUser = response.user;
                    Debug.Log($"User authenticated: {currentUser.username}, Balance: ${currentUser.current_balance}");
                    OnUserAuthenticated?.Invoke(currentUser);
                    StartGameSession();
                }
                else
                {
                    OnError?.Invoke(response.message);
                }
            }));
        }

        #endregion

        #region Session Management

        public void StartGameSession()
        {
            if (currentUser != null && !isSessionActive)
            {
                StartCoroutine(StartSessionCoroutine());
            }
        }

        private IEnumerator StartSessionCoroutine()
        {
            SessionRequest sessionData = new SessionRequest
            {
                user_id = currentUser.id
            };

            yield return StartCoroutine(SendPostRequest<SessionResponse>("start_session.php", sessionData, (response) =>
            {
                if (response.success)
                {
                    currentSessionId = response.session_id;
                    isSessionActive = true;
                    Debug.Log($"Session started: {currentSessionId}");
                }
                else
                {
                    OnError?.Invoke(response.message);
                }
            }));
        }

        public void EndGameSession(bool raceCompleted = false)
        {
            if (isSessionActive && currentSessionId > 0)
            {
                StartCoroutine(EndSessionCoroutine(raceCompleted));
            }
        }

        private IEnumerator EndSessionCoroutine(bool raceCompleted)
        {
            EndSessionRequest endData = new EndSessionRequest
            {
                session_id = currentSessionId,
                race_completed = raceCompleted
            };

            yield return StartCoroutine(SendPostRequest<EndSessionResponse>("end_session.php", endData, (response) =>
            {
                if (response.success)
                {
                    Debug.Log($"Session ended. Earnings: ${response.session_data.earnings}");
                    OnSessionEnded?.Invoke(response.session_data);
                    
                    // Update user balance
                    currentUser.current_balance += response.session_data.earnings;
                    currentUser.total_earned += response.session_data.earnings;
                    
                    isSessionActive = false;
                    currentSessionId = 0;
                }
                else
                {
                    OnError?.Invoke(response.message);
                }
            }));
        }

        #endregion

        #region AdMob Integration

        public void RecordAdMobRevenue(string adType, float revenue, string adUnitId = null)
        {
            if (currentUser != null)
            {
                StartCoroutine(RecordAdMobCoroutine(adType, revenue, adUnitId));
            }
        }

        private IEnumerator RecordAdMobCoroutine(string adType, float revenue, string adUnitId)
        {
            AdMobRequest adData = new AdMobRequest
            {
                user_id = currentUser.id,
                ad_type = adType,
                revenue = revenue,
                ad_unit_id = adUnitId
            };

            yield return StartCoroutine(SendPostRequest<ApiResponse>("record_admob.php", adData, (response) =>
            {
                if (response.success)
                {
                    Debug.Log($"AdMob revenue recorded: {adType} - ${revenue}");
                    
                    // Update user AdMob earnings (70% revenue share)
                    float userShare = revenue * 0.7f;
                    currentUser.current_balance += userShare;
                    currentUser.admob_earnings += userShare;
                }
                else
                {
                    OnError?.Invoke(response.message);
                }
            }));
        }

        public void OnRewardedAdComplete(float estimatedRevenue = 0.01f)
        {
            RecordAdMobRevenue("rewarded", estimatedRevenue);
        }

        public void OnInterstitialAdComplete(float estimatedRevenue = 0.005f)
        {
            RecordAdMobRevenue("interstitial", estimatedRevenue);
        }

        public void OnBannerAdImpression(float estimatedRevenue = 0.001f)
        {
            RecordAdMobRevenue("banner", estimatedRevenue);
        }

        #endregion

        #region Withdrawal System

        public void RequestWithdrawal(float amount, string paymentType, string paymentAddress)
        {
            if (currentUser == null)
            {
                OnError?.Invoke("User not authenticated");
                return;
            }

            if (amount > currentUser.current_balance)
            {
                OnError?.Invoke("Insufficient balance");
                return;
            }

            StartCoroutine(WithdrawalCoroutine(amount, paymentType, paymentAddress));
        }

        private IEnumerator WithdrawalCoroutine(float amount, string paymentType, string paymentAddress)
        {
            WithdrawalRequest withdrawalData = new WithdrawalRequest
            {
                user_id = currentUser.id,
                amount = amount,
                payment_type = paymentType,
                payment_address = paymentAddress
            };

            yield return StartCoroutine(SendPostRequest<ApiResponse>("withdrawal.php", withdrawalData, (response) =>
            {
                if (response.success)
                {
                    Debug.Log("Withdrawal request submitted successfully");
                    // Update user balance (amount is deducted on server)
                    currentUser.current_balance -= amount;
                }
                else
                {
                    OnError?.Invoke(response.message);
                }
            }));
        }

        #endregion

        #region Utility Methods

        private IEnumerator SendPostRequest<T>(string endpoint, object data, System.Action<T> callback) where T : class
        {
            string jsonData = JsonUtility.ToJson(data);
            
            using (UnityWebRequest request = UnityWebRequest.Post(apiBaseUrl + endpoint, jsonData, "application/json"))
            {
                yield return request.SendWebRequest();

                if (request.result == UnityWebRequest.Result.Success)
                {
                    try
                    {
                        T response = JsonUtility.FromJson<T>(request.downloadHandler.text);
                        callback?.Invoke(response);
                    }
                    catch (System.Exception e)
                    {
                        Debug.LogError($"JSON parsing error: {e.Message}");
                        OnError?.Invoke("Response parsing error");
                    }
                }
                else
                {
                    Debug.LogError($"API request failed: {request.error}");
                    OnError?.Invoke($"Network error: {request.error}");
                }
            }
        }

        private IEnumerator SendGetRequest<T>(string endpoint, System.Action<T> callback) where T : class
        {
            using (UnityWebRequest request = UnityWebRequest.Get(apiBaseUrl + endpoint))
            {
                yield return request.SendWebRequest();

                if (request.result == UnityWebRequest.Result.Success)
                {
                    try
                    {
                        T response = JsonUtility.FromJson<T>(request.downloadHandler.text);
                        callback?.Invoke(response);
                    }
                    catch (System.Exception e)
                    {
                        Debug.LogError($"JSON parsing error: {e.Message}");
                        OnError?.Invoke("Response parsing error");
                    }
                }
                else
                {
                    Debug.LogError($"API request failed: {request.error}");
                    OnError?.Invoke($"Network error: {request.error}");
                }
            }
        }

        #endregion

        #region Public Getters

        public string GetFormattedBalance()
        {
            return currentUser != null ? $"${currentUser.current_balance:F2}" : "$0.00";
        }

        public string GetFormattedTotalEarned()
        {
            return currentUser != null ? $"${currentUser.total_earned:F2}" : "$0.00";
        }

        public string GetFormattedPlaytime()
        {
            if (currentUser != null)
            {
                int hours = currentUser.total_playtime_minutes / 60;
                int minutes = currentUser.total_playtime_minutes % 60;
                return $"{hours}h {minutes}m";
            }
            return "0h 0m";
        }

        #endregion
    }
}
