using UnityEngine;
using UnityEngine.UI;
using TMPro;

namespace FierceGame.UI
{
    public class GameUI : MonoBehaviour
    {
        [Header("UI References")]
        [SerializeField] private TextMeshProUGUI balanceText;
        [SerializeField] private TextMeshProUGUI totalEarnedText;
        [SerializeField] private TextMeshProUGUI playtimeText;
        [SerializeField] private TextMeshProUGUI usernameText;
        [SerializeField] private Button withdrawButton;
        [SerializeField] private GameObject withdrawalPanel;
        [SerializeField] private TMP_InputField withdrawAmountInput;
        [SerializeField] private TMP_Dropdown paymentTypeDropdown;
        [SerializeField] private TMP_InputField paymentAddressInput;
        [SerializeField] private Button submitWithdrawalButton;
        [SerializeField] private Button cancelWithdrawalButton;
        [SerializeField] private GameObject loadingPanel;
        [SerializeField] private TextMeshProUGUI statusText;

        [Header("API Reference")]
        [SerializeField] private FierceGameAPI gameAPI;

        void Start()
        {
            if (gameAPI == null)
                gameAPI = FindObjectOfType<FierceGameAPI>();

            SetupEventListeners();
            UpdateUI();
        }

        void SetupEventListeners()
        {
            if (gameAPI != null)
            {
                gameAPI.OnUserAuthenticated.AddListener(OnUserAuthenticated);
                gameAPI.OnSessionEnded.AddListener(OnSessionEnded);
                gameAPI.OnError.AddListener(OnAPIError);
            }

            withdrawButton.onClick.AddListener(ShowWithdrawalPanel);
            submitWithdrawalButton.onClick.AddListener(SubmitWithdrawal);
            cancelWithdrawalButton.onClick.AddListener(HideWithdrawalPanel);
        }

        void OnUserAuthenticated(UserData user)
        {
            UpdateUI();
            ShowStatus($"Welcome, {user.username}!", 3f);
        }

        void OnSessionEnded(SessionData sessionData)
        {
            UpdateUI();
            if (sessionData.earnings > 0)
            {
                ShowStatus($"Session ended! Earned: ${sessionData.earnings:F2}", 5f);
            }
        }

        void OnAPIError(string error)
        {
            ShowStatus($"Error: {error}", 5f);
        }

        void UpdateUI()
        {
            if (gameAPI?.CurrentUser != null)
            {
                var user = gameAPI.CurrentUser;
                balanceText.text = gameAPI.GetFormattedBalance();
                totalEarnedText.text = gameAPI.GetFormattedTotalEarned();
                playtimeText.text = gameAPI.GetFormattedPlaytime();
                usernameText.text = user.username;
                
                // Enable/disable withdrawal button based on balance
                withdrawButton.interactable = user.current_balance >= 5.0f; // Minimum withdrawal
            }
            else
            {
                balanceText.text = "$0.00";
                totalEarnedText.text = "$0.00";
                playtimeText.text = "0h 0m";
                usernameText.text = "Guest";
                withdrawButton.interactable = false;
            }
        }

        void ShowWithdrawalPanel()
        {
            withdrawalPanel.SetActive(true);
            withdrawAmountInput.text = "";
            paymentAddressInput.text = "";
            
            // Set max withdrawal amount
            if (gameAPI?.CurrentUser != null)
            {
                withdrawAmountInput.placeholder.GetComponent<TextMeshProUGUI>().text = 
                    $"Max: ${gameAPI.CurrentUser.current_balance:F2}";
            }
        }

        void HideWithdrawalPanel()
        {
            withdrawalPanel.SetActive(false);
        }

        void SubmitWithdrawal()
        {
            if (gameAPI?.CurrentUser == null) return;

            // Validate input
            if (!float.TryParse(withdrawAmountInput.text, out float amount))
            {
                ShowStatus("Please enter a valid amount", 3f);
                return;
            }

            if (amount < 5.0f)
            {
                ShowStatus("Minimum withdrawal is $5.00", 3f);
                return;
            }

            if (amount > gameAPI.CurrentUser.current_balance)
            {
                ShowStatus("Insufficient balance", 3f);
                return;
            }

            if (string.IsNullOrEmpty(paymentAddressInput.text))
            {
                ShowStatus("Please enter payment address", 3f);
                return;
            }

            // Get payment type
            string paymentType = paymentTypeDropdown.options[paymentTypeDropdown.value].text.ToLower();

            // Submit withdrawal request
            gameAPI.RequestWithdrawal(amount, paymentType, paymentAddressInput.text);
            HideWithdrawalPanel();
            ShowStatus("Withdrawal request submitted!", 5f);
        }

        void ShowStatus(string message, float duration)
        {
            if (statusText != null)
            {
                statusText.text = message;
                statusText.gameObject.SetActive(true);
                
                // Hide after duration
                Invoke(nameof(HideStatus), duration);
            }
        }

        void HideStatus()
        {
            if (statusText != null)
            {
                statusText.gameObject.SetActive(false);
            }
        }

        public void OnRaceCompleted()
        {
            if (gameAPI != null)
            {
                gameAPI.EndGameSession(true);
            }
        }

        public void OnGameExited()
        {
            if (gameAPI != null)
            {
                gameAPI.EndGameSession(false);
            }
        }

        // Call this when showing a rewarded ad
        public void OnRewardedAdWatched()
        {
            if (gameAPI != null)
            {
                gameAPI.OnRewardedAdComplete();
                ShowStatus("Bonus earned from ad!", 3f);
            }
        }

        // Call this when an interstitial ad is shown
        public void OnInterstitialAdShown()
        {
            if (gameAPI != null)
            {
                gameAPI.OnInterstitialAdComplete();
            }
        }
    }
}
